import { Hono } from 'hono'
import { renderer } from './renderer'

const app = new Hono()

app.use(renderer)

// サービス情報
const services = [
  {
    id: 'ayumi-soken',
    title: 'あゆみ所見自動生成システム',
    description: '学生の活動記録から個別の所見を自動生成します。効率的で質の高い所見作成をサポート。',
    url: 'https://ayumi-soken-vercel.vercel.app/',
    icon: 'fas fa-magic',
    color: 'bg-green-500',
    features: ['自動生成', '個別対応', '時間短縮']
  },
  {
    id: 'ayumi-kousei',
    title: 'あゆみ所見等 自動校正システム',
    description: 'あゆみの所見文章を自動で校正・チェックします。誤字脱字から文体まで幅広くサポート。',
    url: 'https://romania0310-art.github.io/kousei-system/',
    icon: 'fas fa-check-circle',
    color: 'bg-blue-500',
    features: ['自動校正', '誤字チェック', '文体調整']
  },
  {
    id: 'ai-proofreading',
    title: 'AI活用！文章自動校正システム',
    description: 'AI技術を活用した高精度な文章校正システム。あらゆる種類の文章に対応。',
    url: 'https://japanese-proofreading-system.onrender.com',
    icon: 'fas fa-robot',
    color: 'bg-purple-500',
    features: ['AI校正', '高精度', '多様な文章対応']
  }
]

app.get('/', (c) => {
  return c.render(
    <div class="container mx-auto px-4 py-8">
      {/* ヘッダー */}
      <header class="text-center mb-12">
        <h1 class="text-4xl md:text-5xl font-bold text-gray-800 mb-4">
          <i class="fas fa-pen-nib text-indigo-600 mr-3"></i>
          文章処理ポータル
        </h1>
        <p class="text-xl text-gray-600 mb-2">あゆみシステム統合ポータル</p>
        <p class="text-gray-500">所見生成・校正システムへの統合アクセス</p>
      </header>

      {/* サービスカード */}
      <div class="grid md:grid-cols-2 lg:grid-cols-3 gap-8 mb-12">
        {services.map(service => (
          <div class="bg-white rounded-xl shadow-lg hover:shadow-xl transition-shadow duration-300 overflow-hidden">
            <div class={`${service.color} p-4 text-white`}>
              <div class="flex items-center mb-2">
                <i class={`${service.icon} text-2xl mr-3`}></i>
                <h3 class="text-lg font-semibold">{service.title}</h3>
              </div>
            </div>
            
            <div class="p-6">
              <p class="text-gray-600 mb-4 text-sm leading-relaxed">{service.description}</p>
              
              <div class="mb-4">
                <h4 class="text-sm font-semibold text-gray-800 mb-2">主な機能:</h4>
                <div class="flex flex-wrap gap-2">
                  {service.features.map(feature => (
                    <span class="px-2 py-1 bg-gray-100 text-gray-700 text-xs rounded-full">
                      {feature}
                    </span>
                  ))}
                </div>
              </div>
              
              <a 
                href={service.url} 
                target="_blank" 
                rel="noopener noreferrer"
                class={`inline-flex items-center justify-center w-full px-4 py-2 ${service.color} text-white rounded-lg hover:opacity-90 transition-opacity font-medium text-sm`}
              >
                <span>サービスにアクセス</span>
                <i class="fas fa-external-link-alt ml-2"></i>
              </a>
            </div>
          </div>
        ))}
      </div>

      {/* 利用ガイド */}
      <div class="bg-white rounded-xl shadow-lg p-8 mb-8">
        <h2 class="text-2xl font-bold text-gray-800 mb-6 flex items-center">
          <i class="fas fa-info-circle text-indigo-600 mr-3"></i>
          利用ガイド
        </h2>
        
        <div class="grid md:grid-cols-2 gap-8">
          <div>
            <h3 class="text-lg font-semibold text-gray-800 mb-3">💭 所見作成の流れ</h3>
            <ol class="space-y-2 text-gray-600">
              <li class="flex items-start">
                <span class="bg-indigo-600 text-white rounded-full w-5 h-5 flex items-center justify-center text-xs mr-3 mt-0.5">1</span>
                <span class="text-sm">あゆみ所見自動生成システムで下書きを作成</span>
              </li>
              <li class="flex items-start">
                <span class="bg-indigo-600 text-white rounded-full w-5 h-5 flex items-center justify-center text-xs mr-3 mt-0.5">2</span>
                <span class="text-sm">あゆみ所見校正システムで内容をチェック・調整</span>
              </li>
              <li class="flex items-start">
                <span class="bg-indigo-600 text-white rounded-full w-5 h-5 flex items-center justify-center text-xs mr-3 mt-0.5">3</span>
                <span class="text-sm">必要に応じてAI校正システムで最終確認</span>
              </li>
            </ol>
          </div>
          
          <div>
            <h3 class="text-lg font-semibold text-gray-800 mb-3">⚙️ システムの特徴</h3>
            <ul class="space-y-2 text-gray-600">
              <li class="flex items-start">
                <i class="fas fa-check text-green-500 mr-3 mt-1"></i>
                <span class="text-sm">学校業務に特化した所見生成・校正</span>
              </li>
              <li class="flex items-start">
                <i class="fas fa-check text-green-500 mr-3 mt-1"></i>
                <span class="text-sm">AI技術による高精度な文章処理</span>
              </li>
              <li class="flex items-start">
                <i class="fas fa-check text-green-500 mr-3 mt-1"></i>
                <span class="text-sm">作業時間の大幅な短縮</span>
              </li>
              <li class="flex items-start">
                <i class="fas fa-check text-green-500 mr-3 mt-1"></i>
                <span class="text-sm">質の高い文章作成サポート</span>
              </li>
            </ul>
          </div>
        </div>
      </div>

      {/* フッター */}
      <footer class="text-center text-gray-500 text-sm">
        <p>&copy; 2024 あゆみシステム文章処理ポータル | 教育現場の効率化をサポート</p>
      </footer>
    </div>,
    { title: '文章処理ポータル | あゆみシステム' }
  )
})

// API エンドポイント（将来的な拡張用）
app.get('/api/services', (c) => {
  return c.json({
    services: services.map(service => ({
      id: service.id,
      title: service.title,
      description: service.description,
      url: service.url,
      features: service.features
    }))
  })
})

export default app
